<?php

namespace App\Http\Controllers;

use App\Models\DetallePreventa;
use App\Models\Preventa;
use App\Models\Producto;
use App\Models\TraspasoConfig;
use App\Models\Ubicacion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
class PreventaController extends Controller
{
 
    public function index(Request $request)
{
    // Obtener el tipo de cambio desde la tabla TraspasoConfig
    $tipoCambioConfig = TraspasoConfig::first();
    $tipoCambio = $tipoCambioConfig->tipo_cambio ?? 1; // Valor predeterminado: 1

    // Cargar la consulta con relaciones
    $query = Preventa::with(['usuario', 'tienda', 'detalles.producto']);

    // Filtro por búsqueda de cliente
    if ($request->has('search')) {
        $search = $request->input('search');
        $query->where('nombre_cliente', 'like', '%' . $search . '%');
    }

    // Filtro por fechas
    if ($request->has('filter')) {
        $filter = $request->input('filter');

        switch ($filter) {
            case 'today':
                $query->whereDate('created_at', today());
                break;
            case 'month':
                $query->whereMonth('created_at', now()->month);
                break;
            case 'year':
                $query->whereYear('created_at', now()->year);
                break;
            case 'period':
                $startDate = Carbon::parse($request->input('start_date'));
                $endDate = Carbon::parse($request->input('end_date'));
                $query->whereBetween('created_at', [$startDate, $endDate]);
                break;
        }
    }

    // Filtros adicionales
    if ($request->filled('specific_date')) {
        $query->whereDate('created_at', $request->input('specific_date'));
    }

    if ($request->filled('specific_month')) {
        $selectedMonth = $request->input('specific_month');
        $query->whereMonth('created_at', $selectedMonth);
    }

    if ($request->filled('start_date') && $request->filled('end_date')) {
        $startDate = Carbon::parse($request->input('start_date'));
        $endDate = Carbon::parse($request->input('end_date'));

        $query->where(function ($query) use ($startDate, $endDate) {
            $query->whereBetween('created_at', [$startDate, $endDate])
                ->orWhereDate('created_at', $startDate)
                ->orWhereDate('created_at', $endDate);
        });
    } elseif ($request->filled('specific_year')) {
        $specificYear = $request->input('specific_year');
        $query->whereYear('created_at', $specificYear);
    }

    // Obtener los resultados paginados
    $preventas = $query->paginate(9)->appends($request->all());

    // Aplicar tipo de cambio a los detalles de preventas
    foreach ($preventas as $preventa) {
        foreach ($preventa->detalles as $detalle) {
            $detalle->precio_con_cambio = $detalle->producto->precio * $tipoCambio;
            $detalle->total_con_cambio = $detalle->cantidad * $detalle->precio_con_cambio;
        }
    }

    // Retornar la vista con los datos y el tipo de cambio
    return view('preventas.index', compact('preventas', 'tipoCambio'));
}

public function create()
{
    $user = Auth::user(); // Obtener el usuario autenticado

    // Verificar el rol del usuario
    if ($user->hasRole('Super administrador')) {
        // Administrador: Puede ver todos los productos con sus ubicaciones
        $productos = Producto::with('ubicacion')->get();
    } elseif ($user->hasRole('Empleado')) {
        // Empleado: Solo puede ver los productos de su ubicación
        $ubicacion_id = $user->empleado->ubicacion_id ?? null;

        if ($ubicacion_id) {
            $productos = Producto::with('ubicacion')
                ->where('ubicacion_id', $ubicacion_id)
                ->get();
        } else {
            return redirect()->back()->with('error', 'No tienes una ubicación asignada.');
        }
    } else {
        // Otros roles: Acceso restringido
        return redirect()->route('home')->with('error', 'No tienes permisos para realizar esta acción.');
    }

    // Obtener todas las ubicaciones para cargar en el select
    $ubicaciones = Ubicacion::all();

    // Obtener el tipo de cambio desde la configuración
    $tipoCambioConfig = TraspasoConfig::first();
    $tipoCambio = $tipoCambioConfig->tipo_cambio ?? 1; // Valor predeterminado: 1

    // Retornar la vista con los datos
    return view('preventas.create', compact('productos', 'ubicaciones', 'tipoCambio'));
}



    public function store(Request $request)
{
    $request->validate([
        'ubicacion_id' => 'required|exists:ubicaciones,id',
        'nombre_cliente' => 'required|string',
        'tel_cliente' => 'required|string',
        'productos' => 'required|array|min:1',
        'productos.*.cantidad' => 'required|integer|min:1',
        'productos.*.precio' => 'required|numeric',
        'productos.*.tipo_precio' => 'required|string'
    ]);

    DB::transaction(function () use ($request) {
        $preventa = Preventa::create([
            'user_id' => Auth::id(),
            'tienda_id' => $request->ubicacion_id,
            'nombre_cliente' => $request->nombre_cliente,
            'tel_cliente' => $request->tel_cliente,
            'precio_total' => $request->precio_total,
            'cantidad_total' => $request->cantidad_total,
        ]);

        foreach ($request->productos as $producto) {
            $preventa->detalles()->create([
                'producto_id' => $producto['id'],
                'cantidad' => $producto['cantidad'],
                'precio' => $producto['precio'],
                'tipo_precio' => $producto['tipo_precio'],
                'precio_total' => $producto['precio'] * $producto['cantidad'],
            ]);
        }
    });

    return redirect()->route('preventas.index')->with('success', 'Preventa registrada exitosamente.');
}

    

    public function edit($id)
    {
        $preventa = Preventa::find($id);
        $productos = Producto::all();
        $ubicaciones = Ubicacion::all();

        return view('preventas.edit', compact('preventa', 'productos', 'ubicaciones'));
    }

    public function update(Request $request, $id)
    {
      
        $request->validate([
            'producto_id' => 'required|exists:productos,id',
            'tienda_id' => 'required|exists:ubicaciones,id',
            'nombre_cliente' => 'required|string',
            'tel_cliente' => 'required|string',
            'cantidad' => 'required|integer|min:1',
            'precio' => 'required|numeric|min:0',
            'precio_total' => 'required|numeric|min:0',
        ]);

   
        $producto = Producto::find($request->input('producto_id'));

    
        $preventa = Preventa::find($id);
        $preventa->producto_id = $request->input('producto_id');
        $preventa->tienda_id = $request->input('tienda_id');
        $preventa->nombre_cliente = $request->input('nombre_cliente');
        $preventa->tel_cliente = $request->input('tel_cliente');
        $preventa->cantidad = $request->input('cantidad');
        $preventa->precio = $request->input('precio');
        $preventa->precio_total = $request->input('precio_total');
        $preventa->save();

        return redirect()->route('preventas.index')->with('success', 'Preventa actualizada exitosamente.');
    }

    public function destroy($id)
    {
      
        $preventa = Preventa::find($id);
        $preventa->delete();

        return redirect()->route('preventas.index')->with('success', 'Preventa eliminada exitosamente.');
    }
}
